"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = convertAST;

var _core = require("@babel/core");

var _visitorKeys = require("../visitor-keys");

function convertNodes(ast, code) {
  const astTransformVisitor = {
    noScope: true,

    enter(path) {
      const {
        node
      } = path;

      if (node.innerComments) {
        delete node.innerComments;
      }

      if (node.trailingComments) {
        delete node.trailingComments;
      }

      if (node.leadingComments) {
        delete node.leadingComments;
      }
    },

    exit(path) {
      const {
        node
      } = path;

      if (node.extra) {
        delete node.extra;
      }

      if (node != null && node.loc.identifierName) {
        delete node.loc.identifierName;
      }

      if (path.isTypeParameter()) {
        node.type = "Identifier";
        node.typeAnnotation = node.bound;
        delete node.bound;
      }

      if (path.isQualifiedTypeIdentifier()) {
        delete node.id;
      }

      if (path.isObjectTypeProperty()) {
        delete node.key;
      }

      if (path.isObjectTypeIndexer()) {
        delete node.id;
      }

      if (path.isFunctionTypeParam()) {
        delete node.name;
      }

      if (path.isImportDeclaration()) {
        delete node.isType;
      }

      if (path.isTemplateLiteral()) {
        for (let i = 0; i < node.quasis.length; i++) {
          const q = node.quasis[i];
          q.range[0] -= 1;

          if (q.tail) {
            q.range[1] += 1;
          } else {
            q.range[1] += 2;
          }

          q.loc.start.column -= 1;

          if (q.tail) {
            q.loc.end.column += 1;
          } else {
            q.loc.end.column += 2;
          }
        }
      }
    }

  };
  const state = {
    source: code
  };
  const oldVisitorKeys = new Map();

  try {
    for (const [type, visitorKey] of Object.entries(_visitorKeys.conflictTypes)) {
      oldVisitorKeys.set(type, _core.types.VISITOR_KEYS[type]);
      _core.types.VISITOR_KEYS[type] = visitorKey;
    }

    for (const [type, visitorKey] of Object.entries(_visitorKeys.newTypes)) {
      _core.types.VISITOR_KEYS[type] = visitorKey;
    }

    (0, _core.traverse)(ast, astTransformVisitor, null, state);
  } finally {
    for (const type of Object.keys(_visitorKeys.newTypes)) {
      delete _core.types.VISITOR_KEYS[type];
    }

    for (const type of Object.keys(_visitorKeys.conflictTypes)) {
      _core.types.VISITOR_KEYS[type] = oldVisitorKeys.get(type);
    }
  }
}

function convertProgramNode(ast) {
  ast.type = "Program";
  ast.sourceType = ast.program.sourceType;
  ast.body = ast.program.body;
  delete ast.program;
  delete ast.errors;

  if (ast.comments.length) {
    const lastComment = ast.comments[ast.comments.length - 1];

    if (ast.tokens.length) {
      const lastToken = ast.tokens[ast.tokens.length - 1];

      if (lastComment.end > lastToken.end) {
        ast.range[1] = lastToken.end;
        ast.loc.end.line = lastToken.loc.end.line;
        ast.loc.end.column = lastToken.loc.end.column;
      }
    }
  } else {
    if (!ast.tokens.length) {
      ast.loc.start.line = 1;
      ast.loc.end.line = 1;
    }
  }

  if (ast.body && ast.body.length > 0) {
    ast.loc.start.line = ast.body[0].loc.start.line;
    ast.range[0] = ast.body[0].start;
  }
}

function convertAST(ast, code) {
  convertNodes(ast, code);
  convertProgramNode(ast);
}